/*
 * jpg_capture.c  - handle JPEG output (using the Independent JPEG Group's (IJG) libjpeg)
 *
 * Copyright (C) 2002 Benjamin Hummel (benjamin@datamaze.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <SDL/SDL.h>
#include <jpeglib.h>
#include "somedefs.h"
#include "my_profiler.h"

/* these are in main.c */
extern BOOL running;
extern BOOL capturing;
extern int capture_mode;
extern unsigned long cap_param;
extern unsigned long cap_val;
extern char *outputfile;
extern int output_number;
extern BOOL verbose;

extern rgba *render_buf;
extern int width;
extern int height;

/* this is in init_exit.c */
void cleanup (void);

/* implement own error handler */
METHODDEF(void)
my_error_exit (j_common_ptr cinfo)
{
	fprintf (stderr, "Internal JPEG error\n");
	cleanup ();
	exit (1);
}

/* save current render_buf to disk */
void
do_capture (void)
{
	FILE *outfile = 0;
	char *fname = 0;
	int i, j;
	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;

	JSAMPROW row_pointer[1];
#define MAX_WIDTH 2048
	BYTE row_buffer[MAX_WIDTH*3+4];

	if (width > MAX_WIDTH)
	{
		fprintf (stderr, "Row buffer too small.\n");
		fprintf (stderr, "Recompile with higher MAX_WIDTH in jpg_capture.c\n");
		running = FALSE;
		return;
	}
#undef MAX_WIDTH

	/* create fname */
	fname = (char *) malloc (strlen (outputfile) + 20);
	if (fname == 0)
	{
		fprintf (stderr, "Not enough memory!\n");
		running = FALSE; return;
	}
	sprintf (fname, "%s%06d.jpg", outputfile, output_number++);

	/*VERBOSE_PRINTF*/
	if (verbose) printf ("Capturing to file %s ... ", fname);

	outfile = fopen (fname, "wb");
	free (fname);
	if (outfile == 0)
	{
	    fprintf(stderr, "Can not open %s\n", fname);
	    running = FALSE; return;
	}

	cinfo.err = jpeg_std_error(&jerr);
	jerr.error_exit = my_error_exit;

	jpeg_create_compress(&cinfo);
	jpeg_stdio_dest(&cinfo, outfile);

	cinfo.image_width = width;
	cinfo.image_height = height;
	cinfo.input_components = 3;
	cinfo.in_color_space = JCS_RGB;

	jpeg_set_defaults(&cinfo);
	jpeg_set_quality(&cinfo, 90, TRUE);

	jpeg_start_compress(&cinfo, TRUE);

	row_pointer[0] = (JSAMPROW)(&row_buffer);
	for (i = 0; i < height; i++)
	{
		/* fill row */
		unsigned long *s = (unsigned long *)(&render_buf[i*width]);
		BYTE *d = (BYTE *)row_buffer;
		for (j = 0; j < width; j++)
		{
			*((unsigned long *)d) = *s++;
			d += 3;
		}
		jpeg_write_scanlines (&cinfo, row_pointer, 1);
	}

	jpeg_finish_compress(&cinfo);
	fclose (outfile);
	jpeg_destroy_compress(&cinfo);

	VERBOSE_PRINTF ("done\n");
}

/* check if capturing should occur and do it */
void
check_capture (void)
{
	BOOL should_capture = FALSE;

	if (!capturing) return;

	switch (capture_mode)
	{
		case CAP_SINGLE:
			running = FALSE;
			/*continue with CAP_MANUAL*/
		case CAP_MANUAL:
			should_capture = TRUE;
			capturing = FALSE;
			break;
		case CAP_TIME:
			if ((SDL_GetTicks() - cap_val) >= cap_param)
			{
				cap_val = SDL_GetTicks();
				should_capture = TRUE;
			}
			break;
		case CAP_FRAMES:
			if (++cap_val >= cap_param)
			{
				cap_val = 0;
				should_capture = TRUE;
			}
			break;
		default:
			fprintf (stderr, "Internal error: invalid capture mode\n");
			running = FALSE;
	}

	if (should_capture)
	{
		myprofiler_start (8, "  capture_frame");
		do_capture();
		myprofiler_stop (8);
	}
}
