/*
 * main.c  - here is the main function and all global vars
 *
 * Copyright (C) 2002 Benjamin Hummel (benjamin@datamaze.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <SDL/SDL.h>
#include <libdc1394/dc1394_control.h>
#include <libraw1394/raw1394.h>

#include <signal.h>

#include "somedefs.h"
#include "my_profiler.h"

/* define effect names */
char *effect_names[FX_MAX-FX_MIN+1] =
	{ "none", "delta", "sobel", "and", "xor", "colorize", "colinv", "colblend" };

/* global states */
BOOL running          = FALSE;
int  effect           = FX_NONE;
BOOL use_grayscale    = FALSE;
rgba color1           = WHITE;
rgba color2           = BLACK;
BYTE param            = 0x30;
BOOL print_statistics = FALSE;
BOOL nowindow         = FALSE;
BOOL verbose          = FALSE;
BOOL use_dma          = FALSE;

/* states for capturing */
BOOL capturing          = FALSE;
int capture_mode        = CAP_MANUAL;
unsigned long cap_param = 0;
unsigned long cap_val   = 0;
char *outputfile        = 0;
int output_number       = 0;

/* vars used for SDL*/
BOOL fullscreen     = FALSE;
BOOL SDL_loaded     = FALSE;
SDL_Surface *screen = NULL;
SDL_Event event;

/* vars used for raw1394 and dc1394 handling*/
raw1394handle_t handle = 0;
nodeid_t *camera_nodes = 0;
dc1394_cameracapture camera;
BOOL iso_started       = FALSE;
BOOL camera_aquired    = FALSE;
int camera_num         = 0;

/* buffers used */
int width        = 640;
int height       = 480;
rgba *new_buf    = NULL;
rgba *old_buf    = NULL;
rgba *render_buf = NULL;


/* external functions: */

/* these are in basic_control.c */
BOOL parse_commandline (int , char **);
void update_caption (void);
void key_handler (void);

/* these are in init_exit.c */
void cleanup (void);
BOOL setup_buffers (void);
BOOL start_1394 (void);
BOOL start_SDL (void);

/* these are in buffer_work.c */
void toggle_buffers (void);
void fill_buffer (void);
void apply_effect_filter (void);
void display_buffer (void);

/* these are in jpg_capture.c */
void do_capture (void);
void check_capture (void);


void
my_sigfunc (int signal)
{
	static BOOL emergency = FALSE;
	if (emergency)
	{
		printf ("Using emergency exit!\n");
		exit(1);
	}
	else
	{
		printf ("Signal detected.\n");
		running = FALSE;
	}

	emergency = TRUE;
}

int
main (int argc, char **argv)
{
	/* quick and dirty endianess test */
	{
		unsigned long i = 1;
		if ( *((char *)(&i)) != 1 ) /* big endian */
		{
			printf ("Sorry, " APP_NAME " currently only supports little endian machines (e.g. x86)\n");
			printf ("If you want to run it nevertheless, you should delete line %d in main.c\n", __LINE__ + 4);
			printf ("Do not expect it to work correctly in this case.\n\n");
			printf ("If you would like to modify aglaia to support big endian machines send an \n");
			printf ("email to <aglaia@datamaze.de>. \nMost of the code affected should be in buffer_work.c\n");
			return 1;
		}
	}


	/* trap signals */
	if (signal (SIGINT, my_sigfunc) == SIG_ERR)
	{
		fprintf (stderr, "Could not set signal handler\n");
		return 1;
	}

	if (	!parse_commandline (argc, argv) ||
		!setup_buffers() ||
		!start_1394() ||
		!start_SDL())
	{
		cleanup();
		return 1;
	}

	VERBOSE_PRINTF ("Starting iso transmission ... ");
	if (dc1394_start_iso_transmission (handle, camera.node) != DC1394_SUCCESS)
	{
		fprintf(stderr, "Could not start isochronous transmission\n");
		cleanup(); return 1;
	}
	else iso_started = TRUE;
	VERBOSE_PRINTF ("done\n");

	/* initially fill buffers */
	fill_buffer();
	fill_buffer(); /* twice, because first result is usually pretty ugly */

	myprofiler_init();

	/* enter main loop */
	running = TRUE;
	while (running)
	{
		myprofiler_start (0, "main loop");

		/* handle SDL events */
		while (!nowindow && SDL_PollEvent (&event))
		{
			switch (event.type)
			{
				case SDL_QUIT: running = 0; break;
				case SDL_KEYDOWN:
				case SDL_KEYUP: key_handler(); break;
				default: /* don't use a default handler */
			}
		}
		if (!running) break;

		update_caption();
		toggle_buffers();

		myprofiler_start (2, "  fill_buffer");
		fill_buffer();
		myprofiler_stop (2);

		myprofiler_start (4, "  apply_filter");
		apply_effect_filter();
		myprofiler_stop ( 4);

		if (!nowindow)
		{
			myprofiler_start (6, "  display_buffer");
			display_buffer();
			myprofiler_stop (6);
		}

		check_capture();

		myprofiler_stop(0);
	}

	if (print_statistics) myprofiler_print();


	cleanup();
	return 0;
}

